/*
 * linux_console.c - Use stdin/stdout as a console device
 *
 * Copyright (c) 2007 Sascha Hauer <s.hauer@pengutronix.de>, Pengutronix
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <common.h>
#include <driver.h>
#include <init.h>
#include <asm/arch/linux.h>
#include <malloc.h>
#include <console.h>
#include <xfuncs.h>

static void linux_console_putc(struct console_device *cdev, char c)
{
	struct device_d *dev = cdev->dev;
	struct linux_console_data *d = dev->platform_data;

	linux_write(d->stdoutfd, &c, 1);
}

static int linux_console_tstc(struct console_device *cdev)
{
	struct device_d *dev = cdev->dev;
	struct linux_console_data *d = dev->platform_data;

	return linux_tstc(d->stdinfd);
}

static int linux_console_getc(struct console_device *cdev)
{
	struct device_d *dev = cdev->dev;
	struct linux_console_data *d = dev->platform_data;
	static char old_c;
	char c;

	linux_read(d->stdinfd, &c, 1);

	if (old_c == 0x1c && c == 'q')
		panic("^\\q pressed - exiting");

	old_c = c;
	return c;
}

static int linux_console_probe(struct device_d *dev)
{
	struct console_device *cdev;
	struct linux_console_data *data = dev->platform_data;

	cdev = xzalloc(sizeof(struct console_device));
	dev->type_data = cdev;
	cdev->dev = dev;
	cdev->f_caps = data->flags;
	cdev->tstc = linux_console_tstc;
	cdev->putc = linux_console_putc;
	cdev->getc = linux_console_getc;

	console_register(cdev);

	return 0;
}

static struct driver_d linux_console_driver = {
        .name  = "console",
        .probe = linux_console_probe,
        .type  = DEVICE_TYPE_CONSOLE,
};

static int console_init(void)
{
	return register_driver(&linux_console_driver);
}

console_initcall(console_init);

