#ifndef __PLAT_GPIO_H
#define __PLAT_GPIO_H

#include <mach/gpio.h>

/*
 * We handle the GPIOs by banks, each bank covers up to 32 GPIOs with
 * one set of registers. The register offsets are organized below:
 *
 *           GPLR    GPDR    GPSR    GPCR    GRER    GFER    GEDR
 * BANK 0 - 0x0000  0x000C  0x0018  0x0024  0x0030  0x003C  0x0048
 * BANK 1 - 0x0004  0x0010  0x001C  0x0028  0x0034  0x0040  0x004C
 * BANK 2 - 0x0008  0x0014  0x0020  0x002C  0x0038  0x0044  0x0050
 *
 * BANK 3 - 0x0100  0x010C  0x0118  0x0124  0x0130  0x013C  0x0148
 * BANK 4 - 0x0104  0x0110  0x011C  0x0128  0x0134  0x0140  0x014C
 * BANK 5 - 0x0108  0x0114  0x0120  0x012C  0x0138  0x0144  0x0150
 *
 * NOTE:
 *   BANK 3 is only available on PXA27x and later processors.
 *   BANK 4 and 5 are only available on PXA935
 */

#define GPIO_BANK(n)	(GPIO_REGS_VIRT + BANK_OFF(n))

#define GPLR_OFFSET	0x00
#define GPDR_OFFSET	0x0C
#define GPSR_OFFSET	0x18
#define GPCR_OFFSET	0x24
#define GRER_OFFSET	0x30
#define GFER_OFFSET	0x3C
#define GEDR_OFFSET	0x48

static inline int gpio_get_value(unsigned gpio)
{
	return GPLR(gpio) & GPIO_bit(gpio);
}

static inline void gpio_set_value(unsigned gpio, int value)
{
	if (value)
		GPSR(gpio) = GPIO_bit(gpio);
	else
		GPCR(gpio) = GPIO_bit(gpio);
}

static inline int gpio_direction_input(unsigned gpio)
{
	if (__gpio_is_inverted(gpio))
		GPDR(gpio) |= GPIO_bit(gpio);
	else
		GPDR(gpio) &= ~GPIO_bit(gpio);
	return 0;
}

static inline int gpio_direction_output(unsigned gpio, int value)
{
	gpio_set_value(gpio, value);
	if (__gpio_is_inverted(gpio))
		GPDR(gpio) &= ~GPIO_bit(gpio);
	else
		GPDR(gpio) |= GPIO_bit(gpio);
	return 0;
}

/* NOTE: some PXAs have fewer on-chip GPIOs (like PXA255, with 85).
 * Those cases currently cause holes in the GPIO number space, the
 * actual number of the last GPIO is recorded by 'pxa_last_gpio'.
 */
extern int pxa_last_gpio;

extern int pxa_init_gpio(int start, int end);

#endif /* __PLAT_GPIO_H */
